<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;

class SetupApplication extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:setup';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Setup the application for first-time installation (for non-technical users)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Setting up your 6FigsTrade application...');
        $this->newLine();

        // Step 1: Create storage link
        $this->info('📁 Setting up file storage...');
        try {
            $publicStoragePath = public_path('storage');
            
            // Remove existing link if it exists
            if (file_exists($publicStoragePath)) {
                if (is_link($publicStoragePath)) {
                    unlink($publicStoragePath);
                } elseif (is_dir($publicStoragePath)) {
                    rmdir($publicStoragePath);
                }
                $this->info('   ✓ Removed existing storage link');
            }
            
            // Create new storage link
            Artisan::call('storage:link');
            $this->info('   ✓ Storage link created successfully');
        } catch (\Exception $e) {
            $this->error('   ✗ Storage link failed: ' . $e->getMessage());
        }

        // Step 2: Clear caches
        $this->info('🧹 Clearing application caches...');
        try {
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            $this->info('   ✓ All caches cleared');
        } catch (\Exception $e) {
            $this->error('   ✗ Cache clearing failed: ' . $e->getMessage());
        }

        // Step 3: Check if .env exists
        $this->info('⚙️  Checking environment configuration...');
        if (!file_exists(base_path('.env'))) {
            if (file_exists(base_path('.env.example'))) {
                copy(base_path('.env.example'), base_path('.env'));
                $this->info('   ✓ Created .env file from .env.example');
                $this->warn('   ⚠️  Please update your .env file with your database and other settings');
            } else {
                $this->error('   ✗ No .env.example file found');
            }
        } else {
            $this->info('   ✓ Environment file exists');
        }

        $this->newLine();
        $this->info('✅ Setup completed successfully!');
        $this->info('🌟 Your 6FigsTrade application is ready to use.');
        $this->newLine();
        $this->comment('📝 Next steps:');
        $this->comment('   1. Update your .env file with database settings');
        $this->comment('   2. Run: php artisan migrate');
        $this->comment('   3. Run: php artisan db:seed (if you have seeders)');
        $this->comment('   4. Visit your website!');
        
        return Command::SUCCESS;
    }
}
