<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;

class CronController extends Controller
{
    /**
     * Run the Laravel scheduler via HTTP request
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function runScheduler(Request $request)
    {
        // Verify the secret token
        $token = $request->input('token') ?? $request->header('X-Cron-Token');
        
        if (!$token || $token !== config('app.cron_secret_token')) {
            Log::warning('Unauthorized cron attempt', [
                'ip' => $request->ip(),
                'token' => $token ? 'provided' : 'missing',
                'time' => now()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        try {
            // Run the scheduler
            Artisan::call('schedule:run');
            $output = Artisan::output();
            
            Log::info('Cron scheduler executed successfully', [
                'time' => now(),
                'output' => $output
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Scheduler executed successfully',
                'time' => now()->toDateTimeString(),
                'output' => $output
            ]);
            
        } catch (\Exception $e) {
            Log::error('Cron scheduler execution failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Scheduler execution failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Run the queue worker for background jobs
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function runQueue(Request $request)
    {
        // Verify the secret token
        $token = $request->input('token') ?? $request->header('X-Cron-Token');
        
        if (!$token || $token !== config('app.cron_secret_token')) {
            Log::warning('Unauthorized queue cron attempt', [
                'ip' => $request->ip(),
                'token' => $token ? 'provided' : 'missing',
                'time' => now()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        try {
            // Process pending jobs (process max 10 jobs or 60 seconds)
            Artisan::call('queue:work', [
                '--stop-when-empty' => true,
                '--max-jobs' => 10,
                '--max-time' => 60,
                '--tries' => 3
            ]);
            
            $output = Artisan::output();
            
            Log::info('Queue worker executed successfully', [
                'time' => now(),
                'output' => $output
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Queue processed successfully',
                'time' => now()->toDateTimeString(),
                'output' => $output
            ]);
            
        } catch (\Exception $e) {
            Log::error('Queue processing failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Queue processing failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Health check endpoint to verify cron setup
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function healthCheck(Request $request)
    {
        // Verify the secret token
        $token = $request->input('token') ?? $request->header('X-Cron-Token');
        
        if (!$token || $token !== config('app.cron_secret_token')) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        try {
            // Check database connection
            \DB::connection()->getPdo();
            $dbStatus = 'connected';
        } catch (\Exception $e) {
            $dbStatus = 'failed: ' . $e->getMessage();
        }

        // Check queue
        try {
            $pendingJobs = \DB::table('jobs')->count();
            $failedJobs = \DB::table('failed_jobs')->count();
            $queueStatus = "Pending: {$pendingJobs}, Failed: {$failedJobs}";
        } catch (\Exception $e) {
            $queueStatus = 'failed: ' . $e->getMessage();
        }

        // Check crypto assets
        try {
            $cryptoCount = \App\Models\CryptoAsset::count();
            $lastUpdate = \App\Models\CryptoAsset::orderBy('last_updated', 'desc')->first();
            $cryptoStatus = "Assets: {$cryptoCount}, Last Update: " . 
                           ($lastUpdate ? $lastUpdate->last_updated->diffForHumans() : 'Never');
        } catch (\Exception $e) {
            $cryptoStatus = 'failed: ' . $e->getMessage();
        }

        return response()->json([
            'success' => true,
            'message' => 'System health check',
            'time' => now()->toDateTimeString(),
            'checks' => [
                'database' => $dbStatus,
                'queue' => $queueStatus,
                'crypto_prices' => $cryptoStatus,
                'timezone' => config('app.timezone'),
                'server_time' => now()->toDateTimeString()
            ]
        ]);
    }
}
