#!/usr/bin/env php
<?php

/**
 * HTTP Cron Setup Helper
 * 
 * This script helps you set up HTTP-based cron jobs for shared hosting
 * Run with: php cron-setup.php
 */

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║         HTTP Cron Setup Helper for Shared Hosting             ║\n";
echo "║                   corepip Application                       ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n";
echo "\n";

// Check if .env file exists
if (!file_exists(__DIR__ . '/.env')) {
    echo "❌ ERROR: .env file not found!\n";
    echo "   Please copy .env.example to .env first.\n";
    exit(1);
}

// Read .env file
$envContent = file_get_contents(__DIR__ . '/.env');

// Check if CRON_SECRET_TOKEN already exists
if (preg_match('/^CRON_SECRET_TOKEN=(.+)$/m', $envContent, $matches)) {
    $existingToken = trim($matches[1]);
    if (!empty($existingToken)) {
        echo "✅ CRON_SECRET_TOKEN already exists in .env\n";
        echo "   Token: {$existingToken}\n\n";
        
        echo "Do you want to generate a new token? (y/N): ";
        $handle = fopen("php://stdin", "r");
        $line = trim(fgets($handle));
        fclose($handle);
        
        if (strtolower($line) !== 'y') {
            $token = $existingToken;
        } else {
            $token = bin2hex(random_bytes(32));
            // Update existing token
            $envContent = preg_replace(
                '/^CRON_SECRET_TOKEN=.*$/m',
                "CRON_SECRET_TOKEN={$token}",
                $envContent
            );
            file_put_contents(__DIR__ . '/.env', $envContent);
            echo "\n✅ New token generated and saved to .env\n";
        }
    } else {
        // Token line exists but empty, generate new one
        $token = bin2hex(random_bytes(32));
        $envContent = preg_replace(
            '/^CRON_SECRET_TOKEN=.*$/m',
            "CRON_SECRET_TOKEN={$token}",
            $envContent
        );
        file_put_contents(__DIR__ . '/.env', $envContent);
        echo "✅ Token generated and saved to .env\n";
    }
} else {
    // Add CRON_SECRET_TOKEN to .env
    $token = bin2hex(random_bytes(32));
    
    // Find APP_URL line and add after it
    if (preg_match('/^APP_URL=.*$/m', $envContent, $matches, PREG_OFFSET_CAPTURE)) {
        $insertPos = $matches[0][1] + strlen($matches[0][0]);
        $envContent = substr_replace(
            $envContent,
            "\n\n# Cron Secret Token (for HTTP-based cron jobs)\nCRON_SECRET_TOKEN={$token}",
            $insertPos,
            0
        );
    } else {
        // If APP_URL not found, append at the end
        $envContent .= "\n\n# Cron Secret Token (for HTTP-based cron jobs)\nCRON_SECRET_TOKEN={$token}\n";
    }
    
    file_put_contents(__DIR__ . '/.env', $envContent);
    echo "✅ Token generated and added to .env\n";
}

// Get APP_URL from .env
if (preg_match('/^APP_URL=(.+)$/m', $envContent, $matches)) {
    $appUrl = trim($matches[1]);
} else {
    $appUrl = 'https://yourdomain.com';
}

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║                    Setup Instructions                          ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n";
echo "\n";

echo "Your Cron URLs (copy these to your cPanel or cron service):\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "1️⃣  SCHEDULER (run every minute):\n";
echo "   {$appUrl}/cron/scheduler?token={$token}\n\n";

echo "2️⃣  QUEUE WORKER (run every minute):\n";
echo "   {$appUrl}/cron/queue?token={$token}\n\n";

echo "3️⃣  HEALTH CHECK (optional - test your setup):\n";
echo "   {$appUrl}/cron/health?token={$token}\n\n";

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "📋 cPanel Cron Job Commands:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "Command 1 (Scheduler):\n";
echo "curl -s \"{$appUrl}/cron/scheduler?token={$token}\" > /dev/null 2>&1\n\n";

echo "Command 2 (Queue Worker):\n";
echo "curl -s \"{$appUrl}/cron/queue?token={$token}\" > /dev/null 2>&1\n\n";

echo "Schedule: */1 * * * * (every minute)\n\n";

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "📝 Next Steps:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "1. Test the health check URL in your browser:\n";
echo "   {$appUrl}/cron/health?token={$token}\n\n";

echo "2. Add the cron jobs to your cPanel:\n";
echo "   - Login to cPanel\n";
echo "   - Find 'Cron Jobs' section\n";
echo "   - Add both commands above\n";
echo "   - Set schedule to: */1 * * * *\n\n";

echo "3. Wait 2 minutes and check if tasks are running:\n";
echo "   - Check storage/logs/laravel.log\n";
echo "   - Verify crypto prices are updating\n";
echo "   - Confirm trades are being processed\n\n";

echo "4. For detailed instructions, see:\n";
echo "   HTTP_CRON_SETUP_GUIDE.md\n\n";

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "⚠️  IMPORTANT: Keep your token secret!\n";
echo "   - Never commit it to git\n";
echo "   - Don't share it publicly\n";
echo "   - Rotate it periodically\n\n";

echo "✅ Setup complete! Your cron token has been configured.\n\n";
