<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Trade;
use App\Enums\TraderType;

class TraderSeeder extends Seeder
{
    public function run(): void
    {
        // Create sample traders
        $traders = [
            [
                'name' => 'John Expert',
                'email' => 'john@trader.com',
                'password' => bcrypt('password'),
                'trader_type' => 'expert',
                'is_trader' => true,
                'copy_trade_fee' => 5.00,
                'trader_description' => 'Professional trader with 5 years of experience',
                'username' => 'john_expert',
                'account_bal' => 10000.00,
                'roi' => 150.00,
                'bonus' => 0.00,
                'ref_bonus' => 0.00,
                'received_signup_bonus' => 0,
                'trade_mode' => 1,
                'status' => 'active',
                'can_withdraw' => 1,
                'can_deposit' => 1,
                'is_admin' => 0,
                'enable_2fa' => 0,
                'pass_two_factor' => 1,
                'has_seen_onboarding' => 1,
                'total_trades' => 0, // Will be updated after trades creation
                'total_wins' => 0,   // Will be updated after trades creation
                'total_losses' => 0,  // Will be updated after trades creation
                'win_rate' => 0.00   // Will be updated after trades creation
            ],
            [
                'name' => 'Sarah Premium',
                'email' => 'sarah@trader.com',
                'password' => bcrypt('password'),
                'trader_type' => 'premium',
                'is_trader' => true,
                'copy_trade_fee' => 3.00,
                'trader_description' => 'Specializing in crypto trends analysis',
                'username' => 'sarah_premium',
                'account_bal' => 8000.00,
                'roi' => 120.00,
                'bonus' => 0.00,
                'ref_bonus' => 0.00,
                'received_signup_bonus' => 0,
                'trade_mode' => 1,
                'status' => 'active',
                'can_withdraw' => 1,
                'can_deposit' => 1,
                'is_admin' => 0,
                'enable_2fa' => 0,
                'pass_two_factor' => 1,
                'has_seen_onboarding' => 1,
                'total_trades' => 0,
                'total_wins' => 0,
                'total_losses' => 0,
                'win_rate' => 0.00
            ],
            [
                'name' => 'Mike VIP',
                'email' => 'mike@trader.com',
                'password' => bcrypt('password'),
                'trader_type' => 'vip',
                'is_trader' => true,
                'copy_trade_fee' => 10.00,
                'trader_description' => 'Top-performing trader with consistent results',
                'username' => 'mike_vip',
                'account_bal' => 25000.00,
                'roi' => 280.00,
                'bonus' => 0.00,
                'ref_bonus' => 0.00,
                'received_signup_bonus' => 0,
                'trade_mode' => 1,
                'status' => 'active',
                'can_withdraw' => 1,
                'can_deposit' => 1,
                'is_admin' => 0,
                'enable_2fa' => 0,
                'pass_two_factor' => 1,
                'has_seen_onboarding' => 1,
                'total_trades' => 0,
                'total_wins' => 0,
                'total_losses' => 0,
                'win_rate' => 0.00
            ],
            [
                'name' => 'Lisa Regular',
                'email' => 'lisa@trader.com',
                'password' => bcrypt('password'),
                'trader_type' => 'regular',
                'is_trader' => true,
                'copy_trade_fee' => 1.00,
                'trader_description' => 'New trader with promising strategy',
                'username' => 'lisa_regular',
                'account_bal' => 5000.00,
                'roi' => 80.00,
                'bonus' => 0.00,
                'ref_bonus' => 0.00,
                'received_signup_bonus' => 0,
                'trade_mode' => 1,
                'status' => 'active',
                'can_withdraw' => 1,
                'can_deposit' => 1,
                'is_admin' => 0,
                'enable_2fa' => 0,
                'pass_two_factor' => 1,
                'has_seen_onboarding' => 1,
                'total_trades' => 0,
                'total_wins' => 0,
                'total_losses' => 0,
                'win_rate' => 0.00
            ]
        ];

        foreach ($traders as $traderData) {
            $trader = User::create($traderData);
            
            // Create sample trades for each trader
            $this->createSampleTrades($trader);
        }
    }

    private function createSampleTrades(User $trader)
    {
        $pairs = ['BTC/USDT', 'ETH/USDT', 'LTC/USDT'];
        $durations = ['1m', '5m', '15m', '30m', '1h'];
        $tradeCount = rand(20, 50); // Random number of trades

        // Calculate win rate based on trader type
        $winRate = match($trader->trader_type) {
            'regular' => 0.55, // 55% win rate
            'premium' => 0.65, // 65% win rate
            'expert' => 0.75,  // 75% win rate
            'vip' => 0.85,     // 85% win rate
            default => 0.50
        };

        $totalProfit = 0;
        for ($i = 0; $i < $tradeCount; $i++) {
            $isWin = (rand(1, 100) / 100) <= $winRate;
            $amount = rand(10, 1000);
            $profitPercentage = 87;
            $profitAmount = $isWin ? ($amount * $profitPercentage / 100) : -$amount;
            $totalProfit += $profitAmount;
            
            $trade = Trade::create([
                'user_id' => $trader->id,
                'pair' => $pairs[array_rand($pairs)],
                'amount' => $amount,
                'type' => rand(0, 1) ? 'Rise' : 'Fall',
                'entry_price' => rand(10000, 50000) / 100,
                'profit_percentage' => $profitPercentage,
                'duration' => $durations[array_rand($durations)],
                'status' => 'completed',
                'result' => $isWin ? 'win' : 'loss',
                'close_price' => rand(10000, 50000) / 100,
                'profit_amount' => $profitAmount,
                'completed_at' => now()->subMinutes(rand(1, 10000)),
                'created_at' => now()->subMinutes(rand(1, 10000))
            ]);
        }

        // Update trader statistics and balance
        $totalTrades = Trade::where('user_id', $trader->id)->count();
        $totalWins = Trade::where('user_id', $trader->id)->where('result', 'win')->count();
        $totalLosses = Trade::where('user_id', $trader->id)->where('result', 'loss')->count();
        
        $trader->update([
            'total_trades' => $totalTrades,
            'total_wins' => $totalWins,
            'total_losses' => $totalLosses,
            'win_rate' => ($totalWins / $totalTrades) * 100,
            'account_bal' => $trader->account_bal + $totalProfit,
            'roi' => ($totalProfit / $trader->account_bal) * 100
        ]);
    }
}